const elements = {
    // 模式按钮和容器
    rgbModeBtn: document.getElementById('rgb-mode'),
    cmykModeBtn: document.getElementById('cmyk-mode'),
    eyedropperBtn: document.getElementById('eyedropper-btn'),
    rgbControls: document.getElementById('rgb-controls'),
    cmykControls: document.getElementById('cmyk-controls'),
    
    // 显示元素
    colorDisplay: document.getElementById('color-display'),
    hexValue: document.getElementById('hex-value'),
    modeValues: document.getElementById('mode-values'),
    paletteContainer: document.getElementById('palette-container'),
    toWhiteContainer: document.getElementById('to-white-container'),
    toBlackContainer: document.getElementById('to-black-container'),
    eyedropperSupportNote: document.getElementById('eyedropper-support'),
    
    // RGB控件
    redSlider: document.getElementById('red'),
    greenSlider: document.getElementById('green'),
    blueSlider: document.getElementById('blue'),
    redValue: document.getElementById('red-value'),
    greenValue: document.getElementById('green-value'),
    blueValue: document.getElementById('blue-value'),
    redValueInput: document.getElementById('red-value-input'),
    greenValueInput: document.getElementById('green-value-input'),
    blueValueInput: document.getElementById('blue-value-input'),
    
    // CMYK控件
    cyanSlider: document.getElementById('cyan'),
    magentaSlider: document.getElementById('magenta'),
    yellowSlider: document.getElementById('yellow'),
    blackSlider: document.getElementById('black'),
    cyanValue: document.getElementById('cyan-value'),
    magentaValue: document.getElementById('magenta-value'),
    yellowValue: document.getElementById('yellow-value'),
    blackValue: document.getElementById('black-value'),
    cyanValueInput: document.getElementById('cyan-value-input'),
    magentaValueInput: document.getElementById('magenta-value-input'),
    yellowValueInput: document.getElementById('yellow-value-input'),
    blackValueInput: document.getElementById('black-value-input')
};

// 颜色模式和状态
let currentMode = 'rgb';
let currentHex = '#000000';
let eyedropperSupported = false;
let lastProcessedColor = null; // 用于防止重复处理相同颜色
let animationFrameId = null; // 用于取消未执行的动画帧

// 初始化
function init() {
    // 检查DOM元素是否存在
    if (!validateElements()) {
        console.error('缺少必要的DOM元素，颜色选择器无法初始化');
        return;
    }
    
    checkEyeDropperSupport();
    updateColorFromRGB();
    setupEventListeners();
}

// 验证必要的DOM元素是否存在
function validateElements() {
    const requiredElements = [
        'rgbModeBtn', 'cmykModeBtn', 'colorDisplay', 
        'hexValue', 'modeValues', 'paletteContainer'
    ];
    
    return requiredElements.every(el => elements[el] !== null);
}

// 检查EyeDropper API支持
function checkEyeDropperSupport() {
    if ('EyeDropper' in window) {
        eyedropperSupported = true;
        elements.eyedropperSupportNote.textContent = "取色功能：点击按钮后，在屏幕任意位置点击可获取颜色";
    } else {
        eyedropperSupported = false;
        elements.eyedropperBtn.disabled = true;
        elements.eyedropperBtn.style.opacity = "0.6";
        elements.eyedropperBtn.style.cursor = "not-allowed";
        elements.eyedropperSupportNote.textContent = "您的浏览器不支持取色功能，请使用最新版Chrome、Edge或Opera浏览器";
    }
}

// 设置事件监听器
function setupEventListeners() {
    // 模式切换按钮
    elements.rgbModeBtn.addEventListener('click', () => switchMode('rgb'));
    elements.cmykModeBtn.addEventListener('click', () => switchMode('cmyk'));
    
    // 取色按钮事件
    elements.eyedropperBtn.addEventListener('click', startColorPick);
    
    // 复制功能事件监听
    elements.hexValue.addEventListener('click', () => copyToClipboard(currentHex));
    elements.colorDisplay.addEventListener('click', () => copyToClipboard(currentHex));
    
    // RGB滑块处理
    ['red', 'green', 'blue'].forEach(color => {
        elements[`${color}Slider`].addEventListener('input', function() {
            const value = this.value;
            elements[`${color}Value`].textContent = value;
            elements[`${color}ValueInput`].value = value;
            updateColorFromRGB();
        });
    });
    
    // RGB输入框处理
    ['red', 'green', 'blue'].forEach(color => {
        elements[`${color}ValueInput`].addEventListener('input', function() {
            const value = clamp(this.value, 0, 255);
            this.value = value;
            elements[`${color}Slider`].value = value;
            elements[`${color}Value`].textContent = value;
            updateColorFromRGB();
        });
    });
    
    // CMYK滑块处理
    ['cyan', 'magenta', 'yellow', 'black'].forEach(color => {
        elements[`${color}Slider`].addEventListener('input', function() {
            const value = this.value;
            elements[`${color}Value`].textContent = value;
            elements[`${color}ValueInput`].value = value;
            updateColorFromCMYK();
        });
    });
    
    // CMYK输入框处理
    ['cyan', 'magenta', 'yellow', 'black'].forEach(color => {
        elements[`${color}ValueInput`].addEventListener('input', function() {
            const value = clamp(this.value, 0, 100);
            this.value = value;
            elements[`${color}Slider`].value = value;
            elements[`${color}Value`].textContent = value;
            updateColorFromCMYK();
        });
    });
    
    // 调色板颜色块点击事件委托
    elements.paletteContainer.addEventListener('click', (e) => {
        const colorBlock = e.target.closest('.palette-color');
        if (colorBlock) {
            const hex = colorBlock.getAttribute('data-hex');
            copyToClipboard(hex);
        }
    });
    
    // 调色板颜色块鼠标事件
    elements.paletteContainer.addEventListener('mouseenter', (e) => {
        const colorBlock = e.target.closest('.palette-color');
        if (colorBlock) {
            const valueDisplay = colorBlock.querySelector('.color-value-display');
            if (valueDisplay) {
                valueDisplay.textContent = currentMode === 'rgb' 
                    ? `RGB: ${colorBlock.getAttribute('data-rgb')}`
                    : `CMYK: ${colorBlock.getAttribute('data-cmyk')}`;
                valueDisplay.style.display = 'block';
            }
        }
    }, true);
    
    elements.paletteContainer.addEventListener('mouseleave', (e) => {
        const colorBlock = e.target.closest('.palette-color');
        if (colorBlock) {
            const valueDisplay = colorBlock.querySelector('.color-value-display');
            if (valueDisplay) {
                valueDisplay.style.display = 'none';
            }
        }
    }, true);
 // 为渐变色块区域添加点击复制功能
elements.toWhiteContainer.addEventListener('click', (e) => {
    const colorBlock = e.target.closest('.palette-color');
    if (colorBlock) {
        const hex = colorBlock.getAttribute('data-hex');
        copyToClipboard(hex);
    }
});

elements.toBlackContainer.addEventListener('click', (e) => {
    const colorBlock = e.target.closest('.palette-color');
    if (colorBlock) {
        const hex = colorBlock.getAttribute('data-hex');
        copyToClipboard(hex);
    }
});

// 为渐变色块区域添加鼠标悬停显示颜色值功能
elements.toWhiteContainer.addEventListener('mouseenter', (e) => {
    const colorBlock = e.target.closest('.palette-color');
    if (colorBlock) {
        const valueDisplay = colorBlock.querySelector('.color-value-display');
        if (valueDisplay) {
            valueDisplay.textContent = currentMode === 'rgb' 
                ? `RGB: ${colorBlock.getAttribute('data-rgb')}`
                : `CMYK: ${colorBlock.getAttribute('data-cmyk')}`;
            valueDisplay.style.display = 'block';
        }
    }
}, true);

elements.toWhiteContainer.addEventListener('mouseleave', (e) => {
    const colorBlock = e.target.closest('.palette-color');
    if (colorBlock) {
        const valueDisplay = colorBlock.querySelector('.color-value-display');
        if (valueDisplay) {
            valueDisplay.style.display = 'none';
        }
    }
}, true);

elements.toBlackContainer.addEventListener('mouseenter', (e) => {
    const colorBlock = e.target.closest('.palette-color');
    if (colorBlock) {
        const valueDisplay = colorBlock.querySelector('.color-value-display');
        if (valueDisplay) {
            valueDisplay.textContent = currentMode === 'rgb' 
                ? `RGB: ${colorBlock.getAttribute('data-rgb')}`
                : `CMYK: ${colorBlock.getAttribute('data-cmyk')}`;
            valueDisplay.style.display = 'block';
        }
    }
}, true);

elements.toBlackContainer.addEventListener('mouseleave', (e) => {
    const colorBlock = e.target.closest('.palette-color');
    if (colorBlock) {
        const valueDisplay = colorBlock.querySelector('.color-value-display');
        if (valueDisplay) {
            valueDisplay.style.display = 'none';
        }
    }
}, true);

}

// 启动取色功能
function startColorPick() {
    if (!eyedropperSupported) return;
    
    const eyeDropper = new EyeDropper();
    
    eyeDropper.open()
        .then(result => {
            const hexColor = result.sRGBHex;
            const rgb = hexToRgb(hexColor);
            
            // 根据当前模式更新对应控件
            if (currentMode === 'rgb') {
                ['red', 'green', 'blue'].forEach((color, i) => {
                    const value = [rgb.r, rgb.g, rgb.b][i];
                    elements[`${color}Slider`].value = value;
                    elements[`${color}Value`].textContent = value;
                    elements[`${color}ValueInput`].value = value;
                });
                updateColorFromRGB();
            } else {
                const cmyk = rgbToCmyk(rgb.r, rgb.g, rgb.b);
                ['cyan', 'magenta', 'yellow', 'black'].forEach((color, i) => {
                    const value = Math.round([cmyk.c, cmyk.m, cmyk.y, cmyk.k][i] * 100);
                    elements[`${color}Slider`].value = value;
                    elements[`${color}Value`].textContent = value;
                    elements[`${color}ValueInput`].value = value;
                });
                updateColorFromCMYK();
            }
            
            showTemporaryMessage("已获取颜色: " + hexColor);
        })
        .catch(err => {
            console.error('取色失败:', err);
            showTemporaryMessage("取色已取消或失败");
        });
}

// 显示临时消息
function showTemporaryMessage(text) {
    // 移除已存在的消息
    const existingMessage = document.querySelector('.temp-message');
    if (existingMessage) {
        existingMessage.remove();
    }
    
    // 创建消息元素
    const message = document.createElement('div');
    message.className = 'temp-message';
    message.textContent = text;
    message.style.position = 'fixed';
    message.style.bottom = '20px';
    message.style.left = '50%';
    message.style.transform = 'translateX(-50%)';
    message.style.backgroundColor = 'rgba(0,0,0,0.8)';
    message.style.color = 'white';
    message.style.padding = '8px 16px';
    message.style.borderRadius = '4px';
    message.style.zIndex = '1000';
    message.style.transition = 'opacity 0.3s ease';
    
    document.body.appendChild(message);
    
    // 3秒后移除
    setTimeout(() => {
        message.style.opacity = '0';
        setTimeout(() => message.remove(), 300);
    }, 3000);
}

// 切换颜色模式
function switchMode(mode) {
    if (mode === currentMode) return;
    
    currentMode = mode;
    
    if (mode === 'rgb') {
        elements.rgbModeBtn.classList.add('active');
        elements.cmykModeBtn.classList.remove('active');
        elements.rgbControls.classList.add('active');
        elements.cmykControls.classList.remove('active');
        
        // 从CMYK转换到RGB
        const c = parseInt(elements.cyanSlider.value) / 100;
        const m = parseInt(elements.magentaSlider.value) / 100;
        const y = parseInt(elements.yellowSlider.value) / 100;
        const k = parseInt(elements.blackSlider.value) / 100;
        
        const rgb = cmykToRgb(c, m, y, k);
        elements.redSlider.value = rgb.r;
        elements.greenSlider.value = rgb.g;
        elements.blueSlider.value = rgb.b;
        elements.redValue.textContent = rgb.r;
        elements.greenValue.textContent = rgb.g;
        elements.blueValue.textContent = rgb.b;
        elements.redValueInput.value = rgb.r;
        elements.greenValueInput.value = rgb.g;
        elements.blueValueInput.value = rgb.b;
        
        updateColorFromRGB();
    } else {
        elements.rgbModeBtn.classList.remove('active');
        elements.cmykModeBtn.classList.add('active');
        elements.rgbControls.classList.remove('active');
        elements.cmykControls.classList.add('active');
        
        // 从RGB转换到CMYK
        const r = parseInt(elements.redSlider.value);
        const g = parseInt(elements.greenSlider.value);
        const b = parseInt(elements.blueSlider.value);
        
        const cmyk = rgbToCmyk(r, g, b);
        elements.cyanSlider.value = Math.round(cmyk.c * 100);
        elements.magentaSlider.value = Math.round(cmyk.m * 100);
        elements.yellowSlider.value = Math.round(cmyk.y * 100);
        elements.blackSlider.value = Math.round(cmyk.k * 100);
        elements.cyanValue.textContent = Math.round(cmyk.c * 100);
        elements.magentaValue.textContent = Math.round(cmyk.m * 100);
        elements.yellowValue.textContent = Math.round(cmyk.y * 100);
        elements.blackValue.textContent = Math.round(cmyk.k * 100);
        elements.cyanValueInput.value = Math.round(cmyk.c * 100);
        elements.magentaValueInput.value = Math.round(cmyk.m * 100);
        elements.yellowValueInput.value = Math.round(cmyk.y * 100);
        elements.blackValueInput.value = Math.round(cmyk.k * 100);
        
        updateColorFromCMYK();
    }
}

// 从RGB更新颜色显示
function updateColorFromRGB() {
    const r = parseInt(elements.redSlider.value);
    const g = parseInt(elements.greenSlider.value);
    const b = parseInt(elements.blueSlider.value);
    const colorKey = `rgb(${r},${g},${b})`;
    
    // 如果颜色没有变化，不进行更新
    if (lastProcessedColor === colorKey) return;
    lastProcessedColor = colorKey;
    
    // 设置颜色显示
    const rgbColor = `rgb(${r}, ${g}, ${b})`;
    elements.colorDisplay.style.backgroundColor = rgbColor;
    
    // 更新十六进制值
    const hex = rgbToHex(r, g, b);
    elements.hexValue.textContent = hex;
    currentHex = hex;
    
    // 更新模式值显示
    elements.modeValues.textContent = `RGB: ${r}, ${g}, ${b}`;
    
    // 取消未执行的动画帧，避免多次渲染
    if (animationFrameId) {
        cancelAnimationFrame(animationFrameId);
    }
    
    // 生成推荐配色方案和渐变
    animationFrameId = requestAnimationFrame(() => {
        generateColorPalette(r, g, b);
        generateGradientToColor(r, g, b, 255, 255, 255, elements.toWhiteContainer);
        generateGradientToColor(r, g, b, 0, 0, 0, elements.toBlackContainer);
    });
}

// 从CMYK更新颜色显示
function updateColorFromCMYK() {
    const c = parseInt(elements.cyanSlider.value) / 100;
    const m = parseInt(elements.magentaSlider.value) / 100;
    const y = parseInt(elements.yellowSlider.value) / 100;
    const k = parseInt(elements.blackSlider.value) / 100;
    const colorKey = `cmyk(${c},${m},${y},${k})`;
    
    // 如果颜色没有变化，不进行更新
    if (lastProcessedColor === colorKey) return;
    lastProcessedColor = colorKey;
    
    // 转换为RGB
    const rgb = cmykToRgb(c, m, y, k);
    
    // 设置颜色显示
    const rgbColor = `rgb(${rgb.r}, ${rgb.g}, ${rgb.b})`;
    elements.colorDisplay.style.backgroundColor = rgbColor;
    
    // 更新十六进制值
    const hex = rgbToHex(rgb.r, rgb.g, rgb.b);
    elements.hexValue.textContent = hex;
    currentHex = hex;
    
    // 更新模式值显示
    elements.modeValues.textContent = `CMYK: ${Math.round(c*100)}%, ${Math.round(m*100)}%, ${Math.round(y*100)}%, ${Math.round(k*100)}%`;
    
    // 取消未执行的动画帧，避免多次渲染
    if (animationFrameId) {
        cancelAnimationFrame(animationFrameId);
    }
    
    // 生成推荐配色方案和渐变
    animationFrameId = requestAnimationFrame(() => {
        generateColorPalette(rgb.r, rgb.g, rgb.b);
        generateGradientToColor(rgb.r, rgb.g, rgb.b, 255, 255, 255, elements.toWhiteContainer);
        generateGradientToColor(rgb.r, rgb.g, rgb.b, 0, 0, 0, elements.toBlackContainer);
    });
}

// 生成推荐配色方案
function generateColorPalette(r, g, b) {
    // 清空现有配色
    elements.paletteContainer.innerHTML = '';
    
    // 生成协调的颜色
    const palette = getHarmoniousColors(r, g, b);
    
    // 使用文档片段减少DOM重绘
    const fragment = document.createDocumentFragment();
    
    // 创建颜色块并添加到片段
    palette.forEach(color => {
        fragment.appendChild(createColorBlock(color));
    });
    
    // 一次性添加到容器
    elements.paletteContainer.appendChild(fragment);
    
    // 生成融合图形展示区
    createFusedPaletteGraphic(palette);
}

// 创建颜色块
function createColorBlock(hexColor) {
    const rgb = hexToRgb(hexColor);
    const cmyk = rgbToCmyk(rgb.r, rgb.g, rgb.b);
    
    const colorBlock = document.createElement('div');
    colorBlock.className = 'palette-color';
    colorBlock.style.backgroundColor = hexColor;
    colorBlock.dataset.hex = hexColor;
    colorBlock.dataset.rgb = `${rgb.r}, ${rgb.g}, ${rgb.b}`;
    colorBlock.dataset.cmyk = `${Math.round(cmyk.c*100)}%, ${Math.round(cmyk.m*100)}%, ${Math.round(cmyk.y*100)}%, ${Math.round(cmyk.k*100)}%`;
    
    // 创建颜色值显示元素
    const valueDisplay = document.createElement('div');
    valueDisplay.className = 'color-value-display';
    valueDisplay.style.display = 'none';
    colorBlock.appendChild(valueDisplay);
    
    return colorBlock;
}

// 生成到目标颜色的渐变
function generateGradientToColor(r1, g1, b1, r2, g2, b2, container) {
    // 清空现有内容
    container.innerHTML = '';
    
    // 生成渐变颜色
    const steps = 11;
    const fragment = document.createDocumentFragment();
    
    for (let i = 0; i <= steps; i++) {
        const ratio = i / steps;
        const r = Math.round(r1 + (r2 - r1) * ratio);
        const g = Math.round(g1 + (g2 - g1) * ratio);
        const b = Math.round(b1 + (b2 - b1) * ratio);
        
        const hex = rgbToHex(r, g, b);
        fragment.appendChild(createColorBlock(hex));
    }
    
    container.appendChild(fragment);
}

// 获取协调颜色函数
function getHarmoniousColors(r, g, b) {
    const hsv = rgbToHsv(r, g, b);
    const palette = [];
    const baseHex = rgbToHex(r, g, b);
    
    // 检查是否为故宫红或黑色，如果是则添加金色推荐
    if (isPalaceRed(r, g, b) || isBlack(r, g, b)) {
        // 添加几种不同明度的金色
        palette.push('#FFD700', '#FFDF00', '#FFFFFF',  '#B8860B');
    }
    
    // 1. 互补色方案 - 色相环180°位置
    const complement = hsvToRgb((hsv.h + 180) % 360, hsv.s, hsv.v);
    palette.push(rgbToHex(complement.r, complement.g, complement.b));
    
    // 2. 类似色方案 - 主色左右15°内
    const analogous1 = hsvToRgb((hsv.h + 15) % 360, hsv.s, hsv.v);
    const analogous2 = hsvToRgb((hsv.h - 15 + 360) % 360, hsv.s, hsv.v);
    palette.push(rgbToHex(analogous1.r, analogous1.g, analogous1.b));
    palette.push(rgbToHex(analogous2.r, analogous2.g, analogous2.b));
    
    // 3. 明暗变化色 - 调整明度生成
    // 更亮的颜色
    const lighter1 = hsvToRgb(hsv.h, hsv.s * 0.9, Math.min(1, hsv.v * 1.2));
    const lighter2 = hsvToRgb(hsv.h, hsv.s * 0.7, Math.min(1, hsv.v * 1.4));
    
    // 更暗的颜色
    const darker1 = hsvToRgb(hsv.h, hsv.s * 1.1, Math.max(0, hsv.v * 0.8));
    const darker2 = hsvToRgb(hsv.h, hsv.s * 1.3, Math.max(0, hsv.v * 0.6));
    
    palette.push(rgbToHex(lighter1.r, lighter1.g, lighter1.b));
    palette.push(rgbToHex(lighter2.r, lighter2.g, lighter2.b));
    palette.push(rgbToHex(darker1.r, darker1.g, darker1.b));
    palette.push(rgbToHex(darker2.r, darker2.g, darker2.b));
    
    // 去重并保持足够对比度
    const uniquePalette = [...new Set(palette)];
    
    // 过滤掉与基础色相同的颜色
    const filteredPalette = uniquePalette.filter(hex => hex !== baseHex);
    
    // 过滤掉对比度过低的颜色
    const contrastFiltered = filteredPalette
        .filter(hex => getContrast(baseHex, hex) > 2.0)
        .slice(0, 12);
    
    // 确保最终返回4个颜色
    return contrastFiltered.slice(0, 4);
}
// 检查是否为故宫红的函数
function isPalaceRed(r, g, b) {
    return r >= 120 && r <= 200 && 
           g >= 20 && g <= 80 && 
           b >= 20 && b <= 80 &&
           r > g * 1.5 && r > b * 1.5;
}

// 检查是否为黑色的函数
function isBlack(r, g, b) {
    return r <= 30 && g <= 30 && b <= 30;
}

// 计算两个颜色的对比度（WCAG标准）
function getContrast(hex1, hex2) {
    const rgb1 = hexToRgb(hex1);
    const rgb2 = hexToRgb(hex2);
    
    // 计算相对亮度
    const luminance1 = getLuminance(rgb1.r, rgb1.g, rgb1.b);
    const luminance2 = getLuminance(rgb2.r, rgb2.g, rgb2.b);
    
    // 计算对比度
    return (Math.max(luminance1, luminance2) + 0.05) / 
           (Math.min(luminance1, luminance2) + 0.05);
}

// 计算颜色的相对亮度（WCAG标准）
function getLuminance(r, g, b) {
    const [r2, g2, b2] = [r, g, b].map(component => {
        const value = component / 255;
        return value <= 0.03928 
            ? value / 12.92 
            : Math.pow((value + 0.055) / 1.055, 2.4);
    });
    return 0.2126 * r2 + 0.7152 * g2 + 0.0722 * b2;
}

// 复制到剪贴板功能
function copyToClipboard(text) {
    // 检查是否支持 navigator.clipboard
    if (navigator.clipboard && window.isSecureContext) {
        // 在安全上下文中使用现代 clipboard API
        navigator.clipboard.writeText(text).then(() => {
            // 显示复制成功的提示消息
            showTemporaryMessage(`已复制: ${text}`);
        }).catch(err => {
            console.error('无法复制: ', err);
            showTemporaryMessage('复制失败！');
        });
    } else {
        // 降级方案：使用传统的 document.execCommand('copy')
        const textArea = document.createElement("textarea");
        textArea.value = text;
        
        // 避免滚动到底部
        textArea.style.top = "0";
        textArea.style.left = "0";
        textArea.style.position = "fixed";
        textArea.style.opacity = "0";
        
        document.body.appendChild(textArea);
        textArea.focus();
        textArea.select();
        
        try {
            const successful = document.execCommand('copy');
            showTemporaryMessage(successful ? `已复制: ${text}` : '复制失败！');
        } catch (err) {
            console.error('无法复制: ', err);
            showTemporaryMessage('复制失败！');
        }
        
        document.body.removeChild(textArea);
    }
}

// 十六进制转RGB
function hexToRgb(hex) {
    // 去掉#号
    hex = hex.replace(/^#/, '');
    
    // 解析RGB值
    const r = parseInt(hex.substring(0, 2), 16);
    const g = parseInt(hex.substring(2, 4), 16);
    const b = parseInt(hex.substring(4, 6), 16);
    
    return { r, g, b };
}

// 颜色转换函数: RGB转十六进制
function rgbToHex(r, g, b) {
    // 确保值在有效范围内
    r = clamp(r, 0, 255);
    g = clamp(g, 0, 255);
    b = clamp(b, 0, 255);
    
    return `#${((1 << 24) + (r << 16) + (g << 8) + b).toString(16).slice(1).toUpperCase()}`;
}

// 颜色转换函数: RGB转CMYK
function rgbToCmyk(r, g, b) {
    // 将RGB值从0-255转换为0-1
    r /= 255;
    g /= 255;
    b /= 255;
    
    // 计算黑色值
    const k = 1 - Math.max(r, g, b);
    
    // 计算青色、品红色、黄色值
    let c, m, y;
    
    if (k === 1) {
        c = m = y = 0;
    } else {
        c = (1 - r - k) / (1 - k);
        m = (1 - g - k) / (1 - k);
        y = (1 - b - k) / (1 - k);
    }
    
    return { c, m, y, k };
}

// 颜色转换函数: CMYK转RGB
function cmykToRgb(c, m, y, k) {
    return {
        r: Math.round(255 * (1 - c) * (1 - k)),
        g: Math.round(255 * (1 - m) * (1 - k)),
        b: Math.round(255 * (1 - y) * (1 - k))
    };
}

// RGB转HSV
function rgbToHsv(r, g, b) {
    r /= 255;
    g /= 255;
    b /= 255;
    
    const max = Math.max(r, g, b);
    const min = Math.min(r, g, b);
    let h = 0; // 初始值
    const v = max;
    const d = max - min;
    const s = max === 0 ? 0 : d / max;
    
    if (max !== min) {
        switch (max) {
            case r: h = (g - b) / d + (g < b ? 6 : 0); break;
            case g: h = (b - r) / d + 2; break;
            case b: h = (r - g) / d + 4; break;
        }
        h *= 60;
    }
    
    return { h, s, v };
}

// HSV转RGB
function hsvToRgb(h, s, v) {
    let r, g, b;
    const i = Math.floor(h / 60);
    const f = h / 60 - i;
    const p = v * (1 - s);
    const q = v * (1 - f * s);
    const t = v * (1 - (1 - f) * s);
    
    switch (i % 6) {
        case 0: r = v; g = t; b = p; break;
        case 1: r = q; g = v; b = p; break;
        case 2: r = p; g = v; b = t; break;
        case 3: r = p; g = q; b = v; break;
        case 4: r = t; g = p; b = v; break;
        case 5: r = v; g = p; b = q; break;
    }
    
    return {
        r: Math.round(r * 255),
        g: Math.round(g * 255),
        b: Math.round(b * 255)
    };
}

// 辅助函数：限制数值在范围内
function clamp(value, min, max) {
    return Math.min(Math.max(parseInt(value) || min, min), max);
}

// 创建融合配色的图形展示
function createFusedPaletteGraphic(paletteColors) {
    // 检查是否已存在融合图形容器，如果存在则移除
    const existingGraphic = document.getElementById('fused-palette-graphic');
    if (existingGraphic) {
        existingGraphic.remove();
    }
    
    // 创建融合图形容器
    const fusedGraphicContainer = document.createElement('div');
    fusedGraphicContainer.id = 'fused-palette-graphic';
    
    // 设置容器样式
    fusedGraphicContainer.style.position = 'fixed';
    fusedGraphicContainer.style.right = '20px';
    fusedGraphicContainer.style.top = '20px';
    fusedGraphicContainer.style.width = '200px';
    fusedGraphicContainer.style.backgroundColor = currentHex;
    fusedGraphicContainer.style.borderRadius = '8px';
    fusedGraphicContainer.style.padding = '15px';
    fusedGraphicContainer.style.boxShadow = '0 4px 12px rgba(0,0,0,0.15)';
    fusedGraphicContainer.style.zIndex = '2';
    fusedGraphicContainer.style.color = getContrastColor(currentHex);
    
    // 创建图形展示区
    const graphicArea = document.createElement('div');
    graphicArea.className = 'fused-graphic-area';
    graphicArea.style.width = '100%';
    graphicArea.style.height = '150px';
    graphicArea.style.position = 'relative';
    graphicArea.style.overflow = 'hidden';
    graphicArea.style.borderRadius = '6px';
    graphicArea.style.marginBottom = '10px';
    
    // 随机选择一种融合图形类型
    const graphicTypes = ['pie'];//写入1个以上则为随机：'pie', 'gradient', 'stripes', 'circles', 'abstract'
    const selectedType = graphicTypes[Math.floor(Math.random() * graphicTypes.length)];
    
    // 根据类型创建图形
    switch(selectedType) {
        case 'gradient':
            createGradientGraphic(graphicArea, paletteColors);
            break;
        case 'pie':
            createPieChartGraphic(graphicArea, paletteColors);
            break;
        case 'stripes':
            createStripesGraphic(graphicArea, paletteColors);
            break;
        case 'circles':
            createCirclesGraphic(graphicArea, paletteColors);
            break;
        case 'abstract':
            createAbstractGraphic(graphicArea, paletteColors);
            break;
    }
    
    // 添加关闭按钮
    const closeButton = document.createElement('button');
    closeButton.textContent = '×';
    closeButton.style.position = 'absolute';
    closeButton.style.top = '5px';
    closeButton.style.right = '10px';
    closeButton.style.background = 'none';
    closeButton.style.border = 'none';
    closeButton.style.fontSize = '18px';
    closeButton.style.cursor = 'pointer';
    closeButton.style.color = getContrastColor(currentHex);
    closeButton.addEventListener('click', () => fusedGraphicContainer.remove());
    
    // 组装元素
    fusedGraphicContainer.appendChild(closeButton);
    fusedGraphicContainer.appendChild(graphicArea);
    
    // 添加到页面
    document.body.appendChild(fusedGraphicContainer);

    // 获取对比色的函数
    function getContrastColor(hexColor) {
        const rgb = hexToRgb(hexColor);
        const luminance = getLuminance(rgb.r, rgb.g, rgb.b);
        return luminance > 0.5 ? '#000000' : '#FFFFFF';
    }
}

// 创建渐变图形
function createGradientGraphic(container, colors) {
    container.style.background = `linear-gradient(45deg, ${colors.slice(0, 3).join(', ')})`;
}

// 创建饼图图形
function createPieChartGraphic(container, colors) {
    const svgNS = "http://www.w3.org/2000/svg";
    const svg = document.createElementNS(svgNS, "svg");
    svg.setAttribute("width", "100%");
    svg.setAttribute("height", "100%");
    svg.setAttribute("viewBox", "0 0 100 100");
    
    const centerX = 50;
    const centerY = 50;
    const radius = 45;
    
    // 如果只有一个颜色，显示为完整圆形
    if (colors.length === 1) {
        const circle = document.createElementNS(svgNS, "circle");
        circle.setAttribute("cx", centerX);
        circle.setAttribute("cy", centerY);
        circle.setAttribute("r", radius);
        circle.setAttribute("fill", colors[0]);
        svg.appendChild(circle);
    } else {
        // 多个颜色的处理逻辑
        let startAngle = 0;
        const anglePerSegment = 360 / Math.min(colors.length,4);
        
        for (let i = 0; i < Math.min(colors.length, 4); i++) {
            const endAngle = startAngle + anglePerSegment;
            
            // 计算路径坐标
            const startRadians = (startAngle - 90) * Math.PI / 180;
            const endRadians = (endAngle - 90) * Math.PI / 180;
            
            const x1 = centerX + radius * Math.cos(startRadians);
            const y1 = centerY + radius * Math.sin(startRadians);
            const x2 = centerX + radius * Math.cos(endRadians);
            const y2 = centerY + radius * Math.sin(endRadians);
            
            // 创建路径
            const path = document.createElementNS(svgNS, "path");
            const largeArcFlag = anglePerSegment > 180 ? 1 : 0;
            path.setAttribute("d", `M ${centerX},${centerY} L ${x1},${y1} A ${radius},${radius} 0 ${largeArcFlag},1 ${x2},${y2} Z`);
            path.setAttribute("fill", colors[i]);
            
            svg.appendChild(path);
            startAngle = endAngle;
        }
    }
    
    container.appendChild(svg);
}

// 创建条纹图形
function createStripesGraphic(container, colors) {
    container.style.display = 'flex';
    container.style.flexDirection = 'column';
    container.style.height = '100%';
    
    const stripeHeight = 100 / Math.min(colors.length, 5);
    
    for (let i = 0; i < Math.min(colors.length, 5); i++) {
        const stripe = document.createElement('div');
        stripe.style.flex = '1';
        stripe.style.backgroundColor = colors[i];
        container.appendChild(stripe);
    }
}

// 创建圆形图形
function createCirclesGraphic(container, colors) {
    container.style.display = 'flex';
    container.style.justifyContent = 'center';
    container.style.alignItems = 'center';
    container.style.backgroundColor = colors[0];
    
    const circleContainer = document.createElement('div');
    circleContainer.style.position = 'relative';
    circleContainer.style.width = '100px';
    circleContainer.style.height = '100px';
    
    for (let i = 0; i < Math.min(colors.length, 4); i++) {
        const circle = document.createElement('div');
        circle.style.position = 'absolute';
        circle.style.borderRadius = '50%';
        circle.style.backgroundColor = colors[i+1] || colors[i];
        
        const size = 100 - (i * 20);
        circle.style.width = `${size}px`;
        circle.style.height = `${size}px`;
        circle.style.top = `${(100 - size) / 2}px`;
        circle.style.left = `${(100 - size) / 2}px`;
        
        circleContainer.appendChild(circle);
    }
    
    container.appendChild(circleContainer);
}

// 创建抽象图形
function createAbstractGraphic(container, colors) {
    container.style.backgroundColor = colors[0];
    container.style.display = 'flex';
    container.style.flexWrap = 'wrap';
    container.style.justifyContent = 'center';
    container.style.alignItems = 'center';
    container.style.gap = '5px';
    container.style.padding = '10px';
    
    for (let i = 1; i < Math.min(colors.length, 6); i++) {
        const shape = document.createElement('div');
        shape.style.backgroundColor = colors[i];
        
        // 随机形状
        const shapes = ['circle', 'square', 'triangle'];
        const selectedShape = shapes[Math.floor(Math.random() * shapes.length)];
        
        switch(selectedShape) {
            case 'circle':
                shape.style.width = '30px';
                shape.style.height = '30px';
                shape.style.borderRadius = '50%';
                break;
            case 'square':
                shape.style.width = '30px';
                shape.style.height = '30px';
                break;
            case 'triangle':
                shape.style.width = '0';
                shape.style.height = '0';
                shape.style.backgroundColor = 'transparent';
                shape.style.borderLeft = '15px solid transparent';
                shape.style.borderRight = '15px solid transparent';
                shape.style.borderBottom = `26px solid ${colors[i]}`;
                break;
        }
        
        container.appendChild(shape);
    }
}

// 初始化应用
document.addEventListener('DOMContentLoaded', init);